/*
CartoTypeBase.h
Copyright 2011-2024 CartoType Ltd.
See www.cartotype.com for more information.
*/

#ifndef CartoTypeBase_h__
#define CartoTypeBase_h__

#import <Foundation/Foundation.h>
#import <CoreGraphics/CGImage.h>
#import <CoreGraphics/CGGeometry.h>
#import <UIKit/UIImage.h>
#import <CartoType/CartoTypeResultCode.h>

/**
\file
Base types for use with the CartoType Objective C class CartoTypeFramework and others.
*/

/** Bit flags used for font styles. */
typedef NS_OPTIONS(NSUInteger,CtFontStyle)
    {
    /** The bit flag used to select bold face. */
    CtFontStyleBold = 1,
    /** The bit flag used to select italics. */
    CtFontStyleItalic = 2,
    /** The bit flag used to select a serif font. */
    CtFontStyleSerif = 4,
    /** The bit flag used to select a cursive font. */
    CtFontStyleCursive = 8,
    /** The bit flag used to select a 'fantasy' font. */
    CtFontStyleFantasy = 16,
    /** The bit flag used to select a monospace font. */
    CtFontStyleMonospace = 32
    };
/** A synonym of CtFontStyle. */
typedef CtFontStyle CartoTypeFontStyle;

/** Flags and constants to tell text searching how to match search terms with found strings. */
typedef NS_OPTIONS(uint32_t,CtStringMatch)
    {
    /** A flag to match all strings for which the search term is a perfect match or a prefix. */
    CtStringMatchPrefixFlag = 1,
    /** A flag to ignore all characters that are not letters, digits or whitespace when matching. */
    CtStringMatchIgnoreSymbolsFlag = 2,
    /** A flag to match accented and unaccented letters. */
    CtStringMatchFoldAccentsFlag = 4,
    /** A flag to allow imperfect matches with a small number of omitted, incorrect or extra characters. */
    CtStringMatchFuzzyFlag = 8,
    /** A flag to fold letter case. */
    CtStringMatchFoldCaseFlag = 16,
    /** A flag to ignore whitespace. */
    CtStringMatchIgnoreWhitespaceFlag = 32,
    
    /** Strings must match exactly. */
    CtStringMatchExact = 0,
    /** Fold case when matching strings. */
    CtStringMatchFoldCase = CtStringMatchFoldCaseFlag,
    /** The search term must be an exact match or a prefix of the found string. */
    CtStringMatchPrefix = CtStringMatchPrefixFlag,
    /** Ignore characters that are not letters, digits or whitespace.  */
    CtStringMatchIgnoreNonAlphanumerics = CtStringMatchIgnoreSymbolsFlag | CtStringMatchIgnoreWhitespaceFlag,
    /** Loose matching: ignore non-alphanumerics and fold accents and case. */
    CtStringMatchLoose = CtStringMatchIgnoreNonAlphanumerics | CtStringMatchFoldAccentsFlag | CtStringMatchFoldCaseFlag,
    /** Allow fuzzy matches: ignore non-alphanumerics, fold accents and allow imperfect matches. */
    CtStringMatchFuzzy = CtStringMatchIgnoreNonAlphanumerics | CtStringMatchFoldAccentsFlag | CtStringMatchFuzzyFlag | CtStringMatchFoldCaseFlag
    };
/** A synonym of CtStringMatch. */
typedef CtStringMatch CartoTypeStringMatch;

/** File types of interest to CartoType. */
typedef NS_ENUM(NSInteger,CtFileType)
    {
    /** PNG (Portable Network Graphics) image files. */
    CtFileTypePng,
    /** JPEG (Joint Photographic Expert Group) image files. */
    CtFileTypeJpeg,
    /** TIFF (Tagged Image File Format) image files. */
    CtFileTypeTiff,
    /** CTM1 (CartoType Map Data Type 1) map data files. */
    CtFileTypeCtm1,
    /** CTMS (CartoType Map Data, Serialized) map data files. */
    CtFileTypeCtms,
    /** KML (Keyhole Markup Language) map data files. */
    CtFileTypeKml,
    /** CTSQL (CartoType SQLite format) map data files. */
    CtFileTypeCtsql,
    /** CTROUTE (CartoType XML route) files. */
    CtFileTypeCtroute,
    /** GPX (GPS Exchange) files. */
    CtFileTypeGpx
    };
/** A synonym of CtFileType. */
typedef CtFileType CartoTypeFileType;

/** Map object types. */
typedef NS_ENUM(NSInteger,CtMapObjectType)
    {
    /** A point in two-dimensional space. */
    CtMapObjectTypePoint,
    
    /** A line in two-dimensional space, consisting of one or more open contours. */
    CtMapObjectTypeLine,
    
    /** A polygon in two-dimensional space, consisting of one or more closed contours. */
    CtMapObjectTypePolygon,
    
    /** Information arranged as an array of values, such as an image or height data. */
    CtMapObjectTypeArray,
    
    /** The total number of map object types. This is not an actual type. */
    CtMapObjectTypeCount,
    
    /** A value to use where a null type is needed. */
    CtMapObjectTypeNone = -1
    };
/** A synonym of CtMapObjectType. */
typedef CtMapObjectType CartoTypeMapObjectType;

/** The type of a CtPathPoint. */
typedef NS_ENUM(NSInteger,CtPointType)
    {
    /** The point is on the curve. */
    CtPointTypeOnCurve = 0,
    /** The point is a control point for a quadratic (conic) Bezier spline. */
    CtPointTypeQuadratic = 1,
    /** The point is a control point for a cubic Bezier spline. */
    CtPointTypeCubic = 2
    };
/** A synonym of CtPointType. */
typedef CtPointType CartoTypePointType;

/** Alignments used for multi-line text. */
typedef NS_ENUM(NSInteger,CtAlign)
    {
    /** Center the text. */
    CtAlignCenter,
    /** Align to the left for left-to-right text, to the right for right-to-left text. */
    CtAlignStandard,
    /** Align to the right for left-to-right text, to the left for right-to-left text. */
    CtAlignReverse,
    /** Align to the left and leave space on the right. */
    CtAlignLeft,
    /** Align to the right and leave space on the left. */
    CtAlignRight
    };
/** A synonym of CtAlign. */
typedef CtAlign CartoTypeAlign;

/** A point on a path, defined using integer values. */
typedef struct
    {
    /** The X coordinate. */
    int32_t x;
    /** The Y coordinate. */
    int32_t y;
    /** The type: on the curve or a control point. */
    CtPointType type;
    } CtPathPoint;
/** A synonym of CtPathPoint. */
typedef CtPathPoint CartoTypePathPoint;

/** A point on a path, defined using double-precision floating point values. */
typedef struct
    {
    /** The X coordinate. */
    double x;
    /** The Y coordinate. */
    double y;
    /** The type: on the curve or a control point. */
    CtPointType type;
    } CtPathPointFP;
/** A synonym of CtPathPointFP. */
typedef CtPathPointFP CartoTypePathPointFP;

/** A point defined using double-precision floating point values. */
typedef struct
    {
    /** The X coordinate. */
    double x;
    /** The Y coordinate. */
    double y;
    } CtPoint;
/** A synonym of CtPoint. */
typedef CtPoint CartoTypePoint;
 
/** An axis-aligned rectangle defined using double-precision floating point values. */
typedef struct
    {
    /** The minimum X coordinate: normally the left edge. */
    double minX;
    /** The minimum Y coordinate: the top if coordinates increase downwards, otherwise the bottom. */
    double minY;
    /** The maximum X coordinate: normally the right edge. */
    double maxX;
    /** The maximum Y coordinate: the bottom if coordinates increase downwards, otherwise the top. */
    double maxY;
    } CtRect;
/** A synonym of CtRect. */
typedef CtRect CartoTypeRect;

/** A point on a height profile. */
@interface CtHeightProfilePoint: NSObject
    /** The position in degrees of longitude (X) and latitude (Y). */
    @property (nonatomic) CtPoint positionInDegrees;
    /** The height above the standard datum (normalized sea level) in meters. */
    @property (nonatomic) int heightInMeters;
    /** The estimated travel time from the start of the route in seconds. */
    @property (nonatomic) double timeInSeconds;
    /** The distance along the route from the start in meters. */
    @property (nonatomic) double distanceInMeters;
@end
/** A synonym of CtHeightProfilePoint */
@compatibility_alias CartoTypeHeightPrpfilePoint CtHeightProfilePoint;

/** A point and a result code. Used by functions that need to return a point but might not succeed. */
typedef struct
    {
    /** The point. */
    CtPoint point;
    /** The result code.  */
    CtResult result;
    } CtPointAndResult;
/** A synonym of CtPointAndResult. */
typedef CtPointAndResult CartoTypePointAndResult;
    
/** A rectangle and a result code. Used by functions that need to return a rectangle but might not succeed. */
typedef struct
    {
    /** The rectangle. */
    CtRect rect;
    /** The result code.  */
    CtResult result;
    } CtRectAndResult;
/** A synonym of CtRectAndResult. */
typedef CtRectAndResult CartoTypeRectAndResult;
   
/** Geometry: a point, polyline or poly-polygon with a coordinate type. */
@class CtGeometry; // forward declaration

/** A geometry object and a result code. Used by functions that need to return a geometry object but might not succeed. */
@interface CtGeometryAndResult: NSObject
    /** The geometry object. */
    @property (nonatomic, strong) CtGeometry* geometry;
    /** The result code. */
    @property (nonatomic) CtResult result;
@end
/** A synonym of CtGeometryAndResult. */
@compatibility_alias CartoTypeGeometryAndResult CtGeometryAndResult;

/** An object ID and a result code. Used by functions that insert an object and need to return its identifier. */
typedef struct
    {
    /** The object ID.  */
    uint64_t objectId;
    /** The result code. */
    CtResult result;
    } CtIdAndResult;
/** A synonym of CtIdAndResult. */
typedef CtIdAndResult CartoTypeIdAndResult;
    
/** An area, a length and a result code. */
typedef struct
    {
    /**The area in square meters.  */
    double area;
    /** The length or perimeter in meters. */
    double length;
    /** The result code. */
    CtResult result;
    } CtAreaLengthAndResult;
/** A synonym of CtAreaLengthAndResult. */
typedef CtAreaLengthAndResult CartoTypeAreaLengthAndResult;

/** Coordinate types used for positions on a map. */
typedef NS_ENUM(NSInteger,CtCoordType)
    {
    /** Longitude (X) and latitude (Y) in degrees. X increases to the east and Y increases to the north. */
    CtCoordTypeDegree,
    /** Pixels on the display: X increases to the right and Y increases downwards. */
    CtCoordTypeDisplay,
    /** A synonym for Display. Pixels on the display: X increases to the right and Y increases downwards. */
    CtCoordTypeScreen = CtCoordTypeDisplay,
    /** Map coordinates: 32nds of projected meters. X increases to the east and Y increases to the north. */
    CtCoordTypeMap,
    /** Map meters: projected meters. X increases to the east and Y increases to the north. */
    CtCoordTypeMapMeter
    };
/** A synonym of CtCoordType. */
typedef CtCoordType CartoTypeCoordType;
    
/** Colors are represented as 32-bit integers holding 8 bits each for the read, green, blue and alpha values. */
typedef uint32_t CtColor NS_TYPED_ENUM;
/** A synonym of CtColor. */
typedef CtColor CartoTypeColor;

/** A value for the color black. */
static CtColor const CtColorBlack = 0xFF000000;
/** A value for the color dark gray. */
static CtColor const CtColorDarkGray = 0xFF555555;
/** A value for the color dark red. */
static CtColor const CtColorDarkRed = 0xFF000080;
/** A value for the color dark green. */
static CtColor const CtColorDarkGreen = 0xFF008000;
/** A value for the color dark yellow. */
static CtColor const CtColorDarkYellow = 0xFF00808;
/** A value for the color dark blue. */
static CtColor const CtColorDarkBlue = 0xFF800000;
/** A value for the color dark magenta. */
static CtColor const CtColorDarkMagenta = 0xFF800080;
/** A value for the color dark cyan. */
static CtColor const CtColorDarkCyan = 0xFF808000;
/** A value for the color red. */
static CtColor const CtColorRed = 0xFF0000FF;
/** A value for the color green. */
static CtColor const CtColorGreen = 0xFF00FF00;
/** A value for the color yellow. */
static CtColor const CtColorYellow = 0xFF00FFFF;
/** A value for the color blue. */
static CtColor const CtColorBlue = 0xFFFF0000;
/** A value for the color magenta. */
static CtColor const CtColorMagenta = 0xFFFF00FF;
/** A value for the color cyan. */
static CtColor const CtColorCyan = 0xFFFFFF00;
/** A value for the color gray. */
static CtColor const CtColorGray = 0xFFAAAAAA;
/** A value for the color white. */
static CtColor const CtColorWhite = 0xFFFFFFFF;
/** A value for the 'null color' transparent black. */
static CtColor const CtColorTransparentBlack = 0x00000000;
    
/**
A rule to modify a certain style or styles.
The colors are blended with colors in a style, using their alpha values as the blend fraction.
*/
@interface CtBlendStyle: NSObject
    /**
    A space or comma separated list of wild-card style names.
    The special name '[legend]' is used for the legend, scale, copyright notice and turn instructions.
    */
    @property (nonatomic) NSString* styles;
    /** The color to be blended with the colors in the style. */
    @property (nonatomic) CtColor mainColor;
    /** If not null, the color to be blended with border colors. */
    @property (nonatomic) CtColor borderColor;
    /** If not null, the color to be blended with text colors. */
    @property (nonatomic) CtColor textColor;
    /** If not null, the color to be blended with text glow colors. */
    @property (nonatomic) CtColor textGlowColor;
    /** If not null, the forced color for icons. */
    @property (nonatomic) CtColor iconColor;
    
    /** Initializes a CtBlendStyle with a C++ blend style: for internal use only. */
    -(id)initWithBlendStyle:(void*)aBlendStyle;
    /** Returns a pointer to the C++ blend style: for internal use only. **/
    -(const void*)blendStyle;
@end
/** A synonym of CtBlendStyle. */
@compatibility_alias CartoTypeBlendStyle CtBlendStyle;

/** Parameters for the function to find an address. */
@interface CtAddress: NSObject
    /** The name or number of the building. */
    @property (nonatomic, strong) NSString* building;
    /** The name of a feature or place of interest. */
    @property (nonatomic, strong) NSString* feature;
    /** The street, road or other highway. */
    @property (nonatomic, strong) NSString* street;
    /** The suburb, neighborhood, quarter or other subdivision of the locality. */
    @property (nonatomic, strong) NSString* subLocality;
    /** The village, town or city. */
    @property (nonatomic, strong) NSString* locality;
    /** The island. */
    @property (nonatomic, strong) NSString* island;
    /** The subsidiary administrative area: district, parish, etc. */
    @property (nonatomic, strong) NSString* subAdminArea;
    /** The administrative area: state, province, etc. */
    @property (nonatomic, strong) NSString* adminArea;
    /** The country. */
    @property (nonatomic, strong) NSString* country;
    /** The postal code. */
    @property (nonatomic, strong) NSString* postCode;
    /**
    Gets the address as a string.
    If aFull is true supplies the administrative divisions
    (county, state, province, etc.) and country.
    */
    -(NSString*)ToString:(bool) aFull;
@end
/** A synonym of CtAddress. */
@compatibility_alias CartoTypeAddress CtAddress;

/** The CartoType path protocol. */
@protocol CtPath <NSObject>
    /** Gets the number of contours (sub-paths) in the path. */
    -(int)contourCount;
    /** Gets the number of points in a certain contour. */
    -(int)pointCount:(int)aContourIndex;
    /** Gets a point, given its contour index and point index. Returns (0,0) if the contour index or point index is out of range. */
    -(CtPathPoint)point:(int)aContourIndex pointIndex:(int)aPointIndex;
@end

/** A synonym of CtPath. */
@protocol CartoTypePath <CtPath>
@end

/** Information returned by the match method of CtMapObject. */
@interface CtMatch: NSObject
    /** True if the text was found.*/
    @property (nonatomic) bool found;
    /** The name of the attribute in which the matched text was found. */
    @property (nonatomic, strong) NSString* key;
    /** The value of the attribute in which the matched text was found. */
    @property (nonatomic, strong) NSString* value;
    /** The start position of the matched text within the value. */
    @property (nonatomic) size_t start;
    /** The end position of the matched text within the value. */
    @property (nonatomic) size_t end;
@end
/** A synonym of CtMatch. */
@compatibility_alias CartoTypeMatch CtMatch;

/** Constructs a 15-bit feature type code from 3 lower-case letters. */
#define MakeFeatureTypeCode(A,B,C) ((((A)-'a'+1) << 10) | (((B)-'a'+1) << 5) | ((C)-'a'+1))

/**
Every map object has a feature type.
The first 32 feature types are used for routable objects and their values are indexes into tables of speeds and weights in route profiles.
Other feature types are constructed from three-letter codes.

All feature type values must fit into 15 bits, and therefore must be in the range 0...32767.
*/
typedef NS_ENUM(NSUInteger,CtFeatureType)
    {
    CtFeatureTypeMotorway = 0,
    CtFeatureTypeMotorwayLink = 1,
    CtFeatureTypeTrunkRoad = 2,
    CtFeatureTypeTrunkLink = 3,
    CtFeatureTypePrimaryRoad = 4,
    CtFeatureTypePrimaryLink = 5,
    CtFeatureTypeSecondaryRoad = 6,
    CtFeatureTypeSecondaryLink = 7,
    CtFeatureTypeTertiaryRoad = 8,
    CtFeatureTypeUnclassifiedRoad = 9,
    CtFeatureTypeResidentialRoad = 10,
    CtFeatureTypeTrack = 11,
    CtFeatureTypeServiceRoad = 12,
    CtFeatureTypePedestrianRoad = 13,
    CtFeatureTypeVehicularFerry = 14,
    CtFeatureTypePassengerFerry = 15,
    CtFeatureTypeLivingStreet = 16,
    CtFeatureTypeCycleway = 17,
    CtFeatureTypePath = 18,
    CtFeatureTypeFootway = 19,
    CtFeatureTypeBridleway = 20,
    CtFeatureTypeSteps = 21,
    CtFeatureTypeOtherRoad = 22,
    CtFeatureTypeUnpavedRoad = 23,
    CtFeatureTypeRailway = 24,
    CtFeatureTypeLightRail = 25,
    CtFeatureTypeSubway = 26,
    CtFeatureTypeAerialWay = 27,
    CtFeatureTypeSkiDownhill = 28,
    CtFeatureTypeSkiNordic = 29,
    CtFeatureTypeWaterway = 30,
    CtFeatureTypeUnknownRoute = 31,

    CtFeatureTypeUnknownNonRoute = 32,

    /*
    Three-letter type codes. Keep these in alphabetic order of type letters to avoid duplicates.
    No routable types should appear among the three-letter codes; all routable objects should be
    given one of the types 0...31.

    The lowest possible three-letter code, aaa, has the value 1057 = 1024 + 32 + 1.
    The highest code, zzz, has the value 27482 = (1024 * 26) + (32 * 26) + 26.
    */
    CtFeatureTypeAdminArea1 = MakeFeatureTypeCode('a','a','a'),
    CtFeatureTypeAdminArea2 = MakeFeatureTypeCode('a','a','b'),
    CtFeatureTypeAdminArea3 = MakeFeatureTypeCode('a','a','c'),
    CtFeatureTypeAdminArea4 = MakeFeatureTypeCode('a','a','d'),
    CtFeatureTypeAdminArea5 = MakeFeatureTypeCode('a','a','e'),
    CtFeatureTypeAdminArea6 = MakeFeatureTypeCode('a','a','f'),
    CtFeatureTypeAdminArea7 = MakeFeatureTypeCode('a','a','g'),
    CtFeatureTypeAdminArea8 = MakeFeatureTypeCode('a','a','h'),
    CtFeatureTypeAdminArea9 = MakeFeatureTypeCode('a','a','i'),
    CtFeatureTypeAdminArea10 = MakeFeatureTypeCode('a','a','j'),
    CtFeatureTypeAddress = MakeFeatureTypeCode('a','d','d'),
    CtFeatureTypeAerodrome = MakeFeatureTypeCode('a','e','r'),
    CtFeatureTypeAirport = MakeFeatureTypeCode('a','i','r'),
    CtFeatureTypeAirportGate = MakeFeatureTypeCode('a','g','t'),
    CtFeatureTypeAirportHoldingPosition = MakeFeatureTypeCode('a','h','p'),
    CtFeatureTypeAllotments = MakeFeatureTypeCode('a','l','l'),
    CtFeatureTypeAlpineHut = MakeFeatureTypeCode('a','l','p'),
    CtFeatureTypeAmbulanceStation = MakeFeatureTypeCode('a','m','b'),
    CtFeatureTypeApron = MakeFeatureTypeCode('a','p','r'),
    CtFeatureTypeAerialWayPylon = MakeFeatureTypeCode('a','p','y'),
    CtFeatureTypeArtwork = MakeFeatureTypeCode('a','r','t'),
    CtFeatureTypeAerialWayStation = MakeFeatureTypeCode('a','s','t'),
    CtFeatureTypeAirTerminal = MakeFeatureTypeCode('a','t','e'),
    CtFeatureTypeAtm = MakeFeatureTypeCode('a','t','m'),
    CtFeatureTypeAttraction = MakeFeatureTypeCode('a','t','t'),
    CtFeatureTypeBar = MakeFeatureTypeCode('b','a','r'),
    CtFeatureTypeBasin = MakeFeatureTypeCode('b','a','s'),
    CtFeatureTypeBay = MakeFeatureTypeCode('b','a','y'),
    CtFeatureTypeBeach = MakeFeatureTypeCode('b','e','a'),
    CtFeatureTypeBeacon = MakeFeatureTypeCode('b','e','c'),
    CtFeatureTypeBedAndBreakfast = MakeFeatureTypeCode('b','e','d'),
    CtFeatureTypeBench = MakeFeatureTypeCode('b','e','n'),
    CtFeatureTypeBank = MakeFeatureTypeCode('b','n','k'),
    CtFeatureTypeBoatyard = MakeFeatureTypeCode('b','o','a'),
    CtFeatureTypeBorough = MakeFeatureTypeCode('b','o','r'),
    CtFeatureTypeBoundary = MakeFeatureTypeCode('b','o','u'),
    CtFeatureTypeBicycleParking = MakeFeatureTypeCode('b','p','k'),
    CtFeatureTypeBicycleRental = MakeFeatureTypeCode('b','r','e'),
    CtFeatureTypeBareRock = MakeFeatureTypeCode('b','r','c'),
    CtFeatureTypeBarracks = MakeFeatureTypeCode('b','r','k'),
    CtFeatureTypeBrownField = MakeFeatureTypeCode('b','r','o'),
    CtFeatureTypeBusStop = MakeFeatureTypeCode('b','s','p'),
    CtFeatureTypeBusStation = MakeFeatureTypeCode('b','s','t'),
    CtFeatureTypeBufferStop = MakeFeatureTypeCode('b','u','f'),
    CtFeatureTypeBuilding = MakeFeatureTypeCode('b','u','i'),
    CtFeatureTypeBunker = MakeFeatureTypeCode('b','u','n'),
    CtFeatureTypeCabin = MakeFeatureTypeCode('c','a','b'),
    CtFeatureTypeCafe = MakeFeatureTypeCode('c','a','f'),
    CtFeatureTypeCampSite = MakeFeatureTypeCode('c','a','m'),
    CtFeatureTypeCanal = MakeFeatureTypeCode('c','a','n'),
    CtFeatureTypeCaveEntrance = MakeFeatureTypeCode('c','a','v'),
    CtFeatureTypeCableCar = MakeFeatureTypeCode('c','c','r'),
    CtFeatureTypeCableDistributionCabinet = MakeFeatureTypeCode('c','d','c'),
    CtFeatureTypeCemetery = MakeFeatureTypeCode('c','e','m'),
    CtFeatureTypeChairLift = MakeFeatureTypeCode('c','h','a'),
    CtFeatureTypeCheckPoint = MakeFeatureTypeCode('c','h','e'),
    CtFeatureTypeChalet = MakeFeatureTypeCode('c','h','l'),
    CtFeatureTypeCivilBoundary = MakeFeatureTypeCode('c','i','b'),
    CtFeatureTypeCinema = MakeFeatureTypeCode('c','i','n'),
    CtFeatureTypeCity = MakeFeatureTypeCode('c','i','t'),
    CtFeatureTypeCliff = MakeFeatureTypeCode('c','l','f'),
    CtFeatureTypeClinic = MakeFeatureTypeCode('c','l','i'),
    CtFeatureTypeCommercial = MakeFeatureTypeCode('c','m','r'),
    CtFeatureTypeCoastline = MakeFeatureTypeCode('c','o','a'),
    CtFeatureTypeCollege = MakeFeatureTypeCode('c','o','l'),
    CtFeatureTypeCommon = MakeFeatureTypeCode('c','o','m'),
    CtFeatureTypeConstruction = MakeFeatureTypeCode('c','n','s'),
    CtFeatureTypeConservation = MakeFeatureTypeCode('c','n','v'),
    CtFeatureTypeContinent = MakeFeatureTypeCode('c','o','n'),
    CtFeatureTypeCounty = MakeFeatureTypeCode('c','o','u'),
    CtFeatureTypeCarPark = MakeFeatureTypeCode('c','p','k'),
    CtFeatureTypeCarRental = MakeFeatureTypeCode('c','r','e'),
    CtFeatureTypeCrossing = MakeFeatureTypeCode('c','r','o'),
    CtFeatureTypeCountry = MakeFeatureTypeCode('c','r','y'),
    CtFeatureTypeCarSharing = MakeFeatureTypeCode('c','s','h'),
    CtFeatureTypeCutLine = MakeFeatureTypeCode('c','u','t'),
    CtFeatureTypeCarWash = MakeFeatureTypeCode('c','w','a'),
    CtFeatureTypeCaravanSite = MakeFeatureTypeCode('c','v','n'),
    CtFeatureTypeCyclingRoute = MakeFeatureTypeCode('c','y','r'),
    CtFeatureTypeDam = MakeFeatureTypeCode('d','a','m'),
    CtFeatureTypeDangerArea = MakeFeatureTypeCode('d','a','n'),
    CtFeatureTypeDentist = MakeFeatureTypeCode('d','e','n'),
    CtFeatureTypeDisused = MakeFeatureTypeCode('d','i','s'),
    CtFeatureTypeDitch = MakeFeatureTypeCode('d','i','t'),
    CtFeatureTypeDock = MakeFeatureTypeCode('d','o','c'),
    CtFeatureTypeDogPark = MakeFeatureTypeCode('d','o','g'),
    CtFeatureTypeDrain = MakeFeatureTypeCode('d','r','a'),
    CtFeatureTypeDragLift = MakeFeatureTypeCode('d','r','g'),
    CtFeatureTypeDoctors = MakeFeatureTypeCode('d','r','s'),
    CtFeatureTypeDistrict = MakeFeatureTypeCode('d','s','t'),
    CtFeatureTypeElectricVehicleCharging = MakeFeatureTypeCode('e','v','c'),
    CtFeatureTypeFarm = MakeFeatureTypeCode('f','a','r'),
    CtFeatureTypeFarmYard = MakeFeatureTypeCode('f','a','y'),
    CtFeatureTypeFell = MakeFeatureTypeCode('f','e','l'),
    CtFeatureTypeFerryTerminal = MakeFeatureTypeCode('f','e','t'),
    CtFeatureTypeFastFood = MakeFeatureTypeCode('f','f','d'),
    CtFeatureTypeFireExtinguisher = MakeFeatureTypeCode('f','i','e'),
    CtFeatureTypeFireFlapper = MakeFeatureTypeCode('f','i','f'),
    CtFeatureTypeFireHose = MakeFeatureTypeCode('f','i','h'),
    CtFeatureTypeFireStation = MakeFeatureTypeCode('f','i','s'),
    CtFeatureTypeFitnessStation = MakeFeatureTypeCode('f','i','t'),
    CtFeatureTypeFireHydrant = MakeFeatureTypeCode('f','i','y'),
    CtFeatureTypeForestry = MakeFeatureTypeCode('f','o','r'),
    CtFeatureTypeFishing = MakeFeatureTypeCode('f','s','h'),
    CtFeatureTypeFuel = MakeFeatureTypeCode('f','u','e'),
    CtFeatureTypeFunicular = MakeFeatureTypeCode('f','u','n'),
    CtFeatureTypeGarages = MakeFeatureTypeCode('g','a','r'),
    CtFeatureTypeGate = MakeFeatureTypeCode('g','a','t'),
    CtFeatureTypeGarden = MakeFeatureTypeCode('g','d','n'),
    CtFeatureTypeGenerator = MakeFeatureTypeCode('g','e','n'),
    CtFeatureTypeGreenHouse = MakeFeatureTypeCode('g','h','o'),
    CtFeatureTypeGlacier = MakeFeatureTypeCode('g','l','a'),
    CtFeatureTypeGolfCourse = MakeFeatureTypeCode('g','o','l'),
    CtFeatureTypeGondola = MakeFeatureTypeCode('g','o','n'),
    CtFeatureTypeGoodsAerialWay = MakeFeatureTypeCode('g','o','o'),
    CtFeatureTypeGrass = MakeFeatureTypeCode('g','r','a'),
    CtFeatureTypeGreenField = MakeFeatureTypeCode('g','r','e'),
    CtFeatureTypeGritBin = MakeFeatureTypeCode('g','r','b'),
    CtFeatureTypeGraveYard = MakeFeatureTypeCode('g','r','y'),
    CtFeatureTypeGuestHouse = MakeFeatureTypeCode('g','u','e'),
    CtFeatureTypeHalt = MakeFeatureTypeCode('h','a','l'),
    CtFeatureTypeHamlet = MakeFeatureTypeCode('h','a','m'),
    CtFeatureTypeHangar = MakeFeatureTypeCode('h','a','n'),
    CtFeatureTypeHeath = MakeFeatureTypeCode('h','e','a'),
    CtFeatureTypeHelipad = MakeFeatureTypeCode('h','e','l'),
    CtFeatureTypeRailwayHalt = MakeFeatureTypeCode('h','l','t'),
    CtFeatureTypeHikingRoute = MakeFeatureTypeCode('h','i','k'),
    CtFeatureTypeHorseRiding = MakeFeatureTypeCode('h','o','r'),
    CtFeatureTypeHospital = MakeFeatureTypeCode('h','o','s'),
    CtFeatureTypeHotel = MakeFeatureTypeCode('h','o','t'),
    CtFeatureTypeHorseRidingRoute = MakeFeatureTypeCode('h','r','r'),
    CtFeatureTypeHostel = MakeFeatureTypeCode('h','s','t'),
    CtFeatureTypeIceRink = MakeFeatureTypeCode('i','c','e'),
    CtFeatureTypeIndustrial = MakeFeatureTypeCode('i','n','d'),
    CtFeatureTypeInformation = MakeFeatureTypeCode('i','n','f'),
    CtFeatureTypeIsland = MakeFeatureTypeCode('i','s','l'),
    CtFeatureTypeIsolatedDwelling = MakeFeatureTypeCode('i','s','o'),
    CtFeatureTypeJunction = MakeFeatureTypeCode('j','c','t'),
    CtFeatureTypeKindergarten = MakeFeatureTypeCode('k','i','n'),
    CtFeatureTypeLandFill = MakeFeatureTypeCode('l','a','n'),
    CtFeatureTypeLand = MakeFeatureTypeCode('l','n','d'),
    CtFeatureTypeLevelCrossing = MakeFeatureTypeCode('l','e','v'),
    CtFeatureTypeLibrary = MakeFeatureTypeCode('l','i','b'),
    CtFeatureTypeLocality = MakeFeatureTypeCode('l','o','c'),
    CtFeatureTypeLockGate = MakeFeatureTypeCode('l','o','k'),
    CtFeatureTypeMaritimeBoundary = MakeFeatureTypeCode('m','a','b'),
    CtFeatureTypeMall = MakeFeatureTypeCode('m','a','l'),
    CtFeatureTypeMarsh = MakeFeatureTypeCode('m','a','r'),
    CtFeatureTypeMeadow = MakeFeatureTypeCode('m','e','a'),
    CtFeatureTypeMilitary = MakeFeatureTypeCode('m','i','l'),
    CtFeatureTypeMarina = MakeFeatureTypeCode('m','n','a'),
    CtFeatureTypeMotel = MakeFeatureTypeCode('m','o','t'),
    CtFeatureTypeMinorPowerLine = MakeFeatureTypeCode('m','p','l'),
    CtFeatureTypeMiniatureGolf = MakeFeatureTypeCode('m','r','g'),
    CtFeatureTypeMiniatureRailway = MakeFeatureTypeCode('m','r','y'),
    CtFeatureTypeMud = MakeFeatureTypeCode('m','u','d'),
    CtFeatureTypeMunicipality = MakeFeatureTypeCode('m','u','n'),
    CtFeatureTypeMuseum = MakeFeatureTypeCode('m','u','s'),
    CtFeatureTypeNatureReserve = MakeFeatureTypeCode('n','a','t'),
    CtFeatureTypeNationalPark = MakeFeatureTypeCode('n','a','p'),
    CtFeatureTypeNavalBase = MakeFeatureTypeCode('n','a','v'),
    CtFeatureTypeNeighborhood = MakeFeatureTypeCode('n','e','i'),
    CtFeatureTypeNursingHome = MakeFeatureTypeCode('n','u','r'),
    CtFeatureTypeOcean = MakeFeatureTypeCode('o','c','e'),
    CtFeatureTypeOrchard = MakeFeatureTypeCode('o','r','c'),
    CtFeatureTypePrecisionApproachPathIndicator = MakeFeatureTypeCode('p','a','p'),
    CtFeatureTypePark = MakeFeatureTypeCode('p','a','r'),
    CtFeatureTypePublicBuilding = MakeFeatureTypeCode('p','b','u'),
    CtFeatureTypePostBox = MakeFeatureTypeCode('p','b','x'),
    CtFeatureTypePostCode = MakeFeatureTypeCode('p','c','o'),
    CtFeatureTypePicnicTable = MakeFeatureTypeCode('p','c','t'),
    CtFeatureTypePeak = MakeFeatureTypeCode('p','e','a'),
    CtFeatureTypePharmacy = MakeFeatureTypeCode('p','h','a'),
    CtFeatureTypePhone = MakeFeatureTypeCode('p','h','o'),
    CtFeatureTypePicnicSite = MakeFeatureTypeCode('p','i','c'),
    CtFeatureTypePier = MakeFeatureTypeCode('p','i','e'),
    CtFeatureTypePipeline = MakeFeatureTypeCode('p','i','p'),
    CtFeatureTypeParkingEntrance = MakeFeatureTypeCode('p','k','e'),
    CtFeatureTypeParking = MakeFeatureTypeCode('p','k','g'),
    CtFeatureTypeParkingSpace = MakeFeatureTypeCode('p','k','s'),
    CtFeatureTypePlantNursery = MakeFeatureTypeCode('p','l','a'),
    CtFeatureTypePlatform = MakeFeatureTypeCode('p','l','f'),
    CtFeatureTypePlaceOfWorship = MakeFeatureTypeCode('p','l','w'),
    CtFeatureTypePlayground = MakeFeatureTypeCode('p','l','y'),
    CtFeatureTypePostOffice = MakeFeatureTypeCode('p','o','f'),
    CtFeatureTypePolice = MakeFeatureTypeCode('p','o','l'),
    CtFeatureTypePosition = MakeFeatureTypeCode('p','o','s'),
    CtFeatureTypeProtectedArea = MakeFeatureTypeCode('p','r','a'),
    CtFeatureTypePowerSubStation = MakeFeatureTypeCode('p','s','s'),
    CtFeatureTypePub = MakeFeatureTypeCode('p','u','b'),
    CtFeatureTypePowerLine = MakeFeatureTypeCode('p','w','l'),
    CtFeatureTypePowerStation = MakeFeatureTypeCode('p','w','s'),
    CtFeatureTypeQuarry = MakeFeatureTypeCode('q','a','r'),
    CtFeatureTypeQuarter = MakeFeatureTypeCode('q','r','t'),
    CtFeatureTypeRange = MakeFeatureTypeCode('r','a','n'),
    CtFeatureTypeRapids = MakeFeatureTypeCode('r','a','p'),
    CtFeatureTypeRecycling = MakeFeatureTypeCode('r','c','y'),
    CtFeatureTypeRecreationGround = MakeFeatureTypeCode('r','e','c'),
    CtFeatureTypeReef = MakeFeatureTypeCode('r','e','e'),
    CtFeatureTypeRegion = MakeFeatureTypeCode('r','e','g'),
    CtFeatureTypeRetail = MakeFeatureTypeCode('r','e','t'),
    CtFeatureTypeRidge = MakeFeatureTypeCode('r','i','j'),
    CtFeatureTypeRiver = MakeFeatureTypeCode('r','i','v'),
    CtFeatureTypeRock = MakeFeatureTypeCode('r','o','c'),
    CtFeatureTypeRoundHouse = MakeFeatureTypeCode('r','o','u'),
    CtFeatureTypeResidentialArea = MakeFeatureTypeCode('r','s','d'),
    CtFeatureTypeRestaurant = MakeFeatureTypeCode('r','s','t'),
    CtFeatureTypeReservoir = MakeFeatureTypeCode('r','s','v'),
    CtFeatureTypeRunway = MakeFeatureTypeCode('r','u','n'),
    CtFeatureTypeRoute = MakeFeatureTypeCode('r','u','t'),
    CtFeatureTypeRiverBank = MakeFeatureTypeCode('r','v','b'),
    CtFeatureTypeSaltPond = MakeFeatureTypeCode('s','a','l'),
    CtFeatureTypeSand = MakeFeatureTypeCode('s','a','n'),
    CtFeatureTypeSauna = MakeFeatureTypeCode('s','a','u'),
    CtFeatureTypeSchool = MakeFeatureTypeCode('s','c','h'),
    CtFeatureTypeScree = MakeFeatureTypeCode('s','c','r'),
    CtFeatureTypeScrub = MakeFeatureTypeCode('s','c','b'),
    CtFeatureTypeSea = MakeFeatureTypeCode('s','e','a'),
    CtFeatureTypeStateEmergencyServiceStation = MakeFeatureTypeCode('s','e','s'),
    CtFeatureTypeShop = MakeFeatureTypeCode('s','h','o'),
    CtFeatureTypeSkiRoute = MakeFeatureTypeCode('s','k','r'),
    CtFeatureTypeSlipway = MakeFeatureTypeCode('s','l','i'),
    CtFeatureTypeSportsCenter = MakeFeatureTypeCode('s','p','o'),
    CtFeatureTypeSportsPitch = MakeFeatureTypeCode('s','p','p'),
    CtFeatureTypeSpring = MakeFeatureTypeCode('s','p','r'),
    CtFeatureTypeSportsTrack = MakeFeatureTypeCode('s','p','t'),
    CtFeatureTypeState = MakeFeatureTypeCode('s','t','a'),
    CtFeatureTypeStadium = MakeFeatureTypeCode('s','t','m'),
    CtFeatureTypeRailwayStation = MakeFeatureTypeCode('s','t','n'),
    CtFeatureTypeStation = MakeFeatureTypeCode('s','t','n'), // deliberate duplicate
    CtFeatureTypeStone = MakeFeatureTypeCode('s','t','o'),
    CtFeatureTypeStopPosition = MakeFeatureTypeCode('s','t','p'),
    CtFeatureTypeStream = MakeFeatureTypeCode('s','t','r'),
    CtFeatureTypeStrait = MakeFeatureTypeCode('s','t','t'),
    CtFeatureTypeSuburb = MakeFeatureTypeCode('s','u','b'),
    CtFeatureTypeSupermarket = MakeFeatureTypeCode('s','u','p'),
    CtFeatureTypeSurveyPoint = MakeFeatureTypeCode('s','u','r'),
    CtFeatureTypeSubwayEntrance = MakeFeatureTypeCode('s','w','e'),
    CtFeatureTypeSwimmingPool = MakeFeatureTypeCode('s','w','i'),
    CtFeatureTypeTank = MakeFeatureTypeCode('t','a','n'),
    CtFeatureTypeTaxi = MakeFeatureTypeCode('t','a','x'),
    CtFeatureTypeTheatre = MakeFeatureTypeCode('t','h','e'),
    CtFeatureTypeThemePark = MakeFeatureTypeCode('t','h','p'),
    CtFeatureTypeToilet = MakeFeatureTypeCode('t','o','i'),
    CtFeatureTypeTown = MakeFeatureTypeCode('t','o','w'),
    CtFeatureTypeTurningCircle = MakeFeatureTypeCode('t','c','i'),
    CtFeatureTypeTurningPoint = MakeFeatureTypeCode('t','p','t'),
    CtFeatureTypeTram = MakeFeatureTypeCode('t','r','a'),
    CtFeatureTypeTree = MakeFeatureTypeCode('t','r','e'),
    CtFeatureTypeTrafficSignals = MakeFeatureTypeCode('t','r','f'),
    CtFeatureTypeTrackPoint = MakeFeatureTypeCode('t','r','p'),
    CtFeatureTypeTreeRow = MakeFeatureTypeCode('t','r','r'),
    CtFeatureTypeTramStop = MakeFeatureTypeCode('t','r','s'),
    CtFeatureTypeTurnTable = MakeFeatureTypeCode('t','u','r'),
    CtFeatureTypeTower = MakeFeatureTypeCode('t','w','r'),
    CtFeatureTypeTaxiway = MakeFeatureTypeCode('t','w','y'),
    CtFeatureTypeUniversity = MakeFeatureTypeCode('u','n','i'),
    CtFeatureTypeVisualApproachSlopeIndicator = MakeFeatureTypeCode('v','a','s'),
    CtFeatureTypeVillageGreen = MakeFeatureTypeCode('v','i','g'),
    CtFeatureTypeVillage = MakeFeatureTypeCode('v','i','l'),
    CtFeatureTypeVineyard = MakeFeatureTypeCode('v','i','n'),
    CtFeatureTypeViewPoint = MakeFeatureTypeCode('v','i','w'),
    CtFeatureTypeVolcano = MakeFeatureTypeCode('v','o','l'),
    CtFeatureTypeWaterfall = MakeFeatureTypeCode('w','a','f'),
    CtFeatureTypeWaterPark = MakeFeatureTypeCode('w','a','p'),
    CtFeatureTypeWater = MakeFeatureTypeCode('w','a','t'),
    CtFeatureTypeWeir = MakeFeatureTypeCode('w','e','r'),
    CtFeatureTypeWetland = MakeFeatureTypeCode('w','e','t'),
    CtFeatureTypeWildernessHut = MakeFeatureTypeCode('w','i','l'),
    CtFeatureTypeWindsock = MakeFeatureTypeCode('w','i','s'),
    CtFeatureTypeWalkingRoute = MakeFeatureTypeCode('w','l','k'),
    CtFeatureTypeWood = MakeFeatureTypeCode('w','o','o'),
    CtFeatureTypeWorks = MakeFeatureTypeCode('w','o','r'),
    CtFeatureTypeWaypoint = MakeFeatureTypeCode('w','p','t'),
    CtFeatureTypeWaterTower = MakeFeatureTypeCode('w','t','t'),
    CtFeatureTypeWaterWell = MakeFeatureTypeCode('w','t','w'),
    CtFeatureTypeZoo = MakeFeatureTypeCode('z','o','o'),

    // legacy types for backward compatibility
    CtFeatureTypeLegacyBridleway = MakeFeatureTypeCode('b','r','i'),
    CtFeatureTypeLegacyCycleway = MakeFeatureTypeCode('c','y','c'),
    CtFeatureTypeLegacyFootway = MakeFeatureTypeCode('f','o','o'),
    CtFeatureTypeLegacyLightRail = MakeFeatureTypeCode('l','i','r'),
    CtFeatureTypeLegacyMonorail = MakeFeatureTypeCode('m','o','n'),
    CtFeatureTypeLegacyNarrowGauge = MakeFeatureTypeCode('n','a','r'),
    CtFeatureTypeLegacyPreservedRailway = MakeFeatureTypeCode('p','r','y'),
    CtFeatureTypeLegacyRailway = MakeFeatureTypeCode('r','l','y'),
    CtFeatureTypeLegacySteps = MakeFeatureTypeCode('s','t','e'),
    CtFeatureTypeLegacySubway = MakeFeatureTypeCode('s','w','y'),
    CtFeatureTypeLegacyTram = MakeFeatureTypeCode('t','r','a'),

    CtFeatureTypeInvalid = INT16_MAX
    };
/** A synonym of CtFeatureType. */
typedef CtFeatureType CartoTypeFeatureType;
    
/** The feature discriminator stored in a FeatureInfo object. */
typedef NS_ENUM(NSUInteger,CtFeatureDiscriminator)
    {
    /** Not a route. */
    CtFeatureDiscriminatorNonRoute = 0,
    /** A route that is a bridge. */
    CtFeatureDiscriminatorBridgeRoute = 1,
    /** A route that is a tunnel. */
    CtFeatureDiscriminatorTunnelRoute = 2,
    /** A route that is neither a tunnel nor a bridge. */
    CtFeatureDiscriminatorRoute = 3
    };
/** A synonym of CtFeatureDiscriminator. */
typedef CtFeatureDiscriminator CartoTypeFeatureDiscriminator;

/**
Feature information for a map object, represented internally as a 32-bit value.

Every map object has a feature info value. Feature info values fall into two categories: route and non-route.

Route values have a feature type in the range 0...31 (CtFeatureTypeMotorway...CtFeatureTypeUnknownRoute); it is used as the index into the speed and bonus
arrays in a route profile. Route values have other information relevant to routing including the speed limit
and access restrictions.

Non-route values have a feature type in the range 32...32767. Non-route feature types, apart from the special values
CtFeatureTypeUnknownNonRoute (32) and CtFeatureTypeInvalid (32767), may be expressed as three-letter mnemonics, which can be constructed using
the MakeFeatureTypeCode macro. Non-route values have an 11-bit sub-type that may be used for fine distinctions such
as settlement rank or boundary type.

Both route and non-route values have a level in the range -8...7, where 0 represents ground level. The level is used as
one of the criteria determining drawing order.
*/
@interface CtFeatureInfo: NSObject
    /** Creates a feature info object of type UnknownNonRoute. */
    -(id)init;
/** Creates a feature info object from a feature type. */
    -(id)initWithType:(CtFeatureType)aFeatureType;
    /**
    Creates a feature info object with a given type and sub-type.
    The sub-type is clamped to the range 0...2047, and is ignored if aType is a route type,
    because routes do not have sub-types.
    */
    -(id)initWithTypeAndSubType:(CtFeatureType)aFeatureType subType:(int)aSubType;
    /**
    Creates a non-route feature info object of type UnknownNonRoute and a given sub-type.
    The sub-type is clamped to the range 0...2047.
    */
    -(id)initWithSubType:(int)aSubType;
    /** The feature discriminator. */
    @property (nonatomic) CtFeatureDiscriminator discriminator;
    /** True if this is a route. */
    @property (nonatomic) bool route;
    /** True if this is a route that is a tunnel. Only route objects can made into tunnels. */
    @property (nonatomic) bool tunnel;
    /** True if this is a route that is a bridge. Only route objects can made into bridges. */
    @property (nonatomic) bool bridge;
    /** The level, which is in the range -8 ... 7. */
    @property (nonatomic) int level;
    /** The feature type. Route objects can only be set to route types, and non-route objects to non-route types. */
    @property (nonatomic) CtFeatureType type;
    /** True if this is a one-way route. Non-route objects cannot be set as one-way.  */
    @property (nonatomic) bool oneWay;
    /** True if this is a one-way route in the direction in which the route is stored. */
    @property (nonatomic) bool oneWayForward;
    /** True if this is a one-way route in the direction opposite to the one in which the route is stored. */
    @property (nonatomic) bool oneWayBackward;
    /** True if this is a two-way route and the rule of the road is to drive on the left. The rule of the road cannot be set for non-route objects. */
    @property (nonatomic) bool driveOnLeft;
    /** Returns true if this is a two-way route and the rule of the road is to drive on the right. The rule of the road cannot be set for non-route objects. */
    @property (nonatomic) bool driveOnRight;
    /** Reverses the one-way direction if this is a one-way route. */
    -(void)reverseOneWayDirection;
    /** True if this is a route that is part of a roundabout. This property cannot be set for non-route objects. */
    @property (nonatomic) bool roundabout;
    /** True if this is a toll route. This property cannot be set for non-route objects.  */
    @property (nonatomic) bool toll;
    /** True if this is a route with a one-way direction opposite to the order of its points. This property cannot be set for non-route objects. */
    @property (nonatomic) bool wrongWay;
    /** The speed limit in kph. A value of zero indicates that no speed limit is known. This property cannot be set for non-route objects. */
    @property (nonatomic) int speedLimit;
    /** The gradient as a number in the range 0...7. Values 0...3 are uphill and 4...7 are downhill. Gradients are stored only in route arcs and not usually in other map objects. This property cannot be set for non-route objects. */
    @property (nonatomic) int gradient;
    /** Sets or clears the vehicle access flags. Throws an exception if this is not a route. */
    -(void)setVehicleAccess:(bool)aValue;
    /** True if this is a route and pedestrian access is allowed. This property cannot be set for non-route objects. */
    @property (nonatomic) bool pedestrianAccess;
    /** True if this is a route and cycle access is allowed. */
    @property (nonatomic) bool cycleAccess;
    /** True if this is a route and motor vehicle access is allowed. This property cannot be set for non-route objects. */
    @property (nonatomic) bool motorVehicleAccess;
    /** True if this is a route and emergency vehicle access is allowed. */
    @property (nonatomic) bool emergencyVehicleAccess;
    /** True if this is a route and there are access restrictions other than those specified by the standard flags. This property cannot be set for non-route objects. */
    @property (nonatomic) bool otherAccessRestricted;
    /** True if this is a route and no normal access is allowed. */
    @property (nonatomic) bool isPrivate;
    /** The sub-type. Always 0 if this is a route. This property cannot be set for route objects. */
    @property (nonatomic) int subType;
    /** Creates a feature info object from a raw value. For internal use only. */
    -(id)initWithRawValue:(uint32_t)aValue;
    /** Returns the raw value. For internal use only. */
    -(uint32_t)rawValue;
@end
/** A synonym of CtFeatureInfo. */
@compatibility_alias CartoTypeFeatureInfo CtFeatureInfo;

/** The number of feature types that are used for routes. These values are also indexes into the speed and bonus arrays in route profiles. */
static const uint32_t CtFeatureInfoRouteTypeCount = 32;
/** The number of different gradients distinguished by a feature info or route profile object. */
static const uint32_t CtFeatureInfoGradientCount = 8;
/** The maximum value for the sub-type of a non-route feature info object. The minimum value is zero. */
static const uint32_t CtFeatureInfoMaxSubType = 2047;

/** A type for flags controlling route access. */
typedef uint32_t CtRouteAccess NS_TYPED_ENUM;

/** The amount by which the route access flags are shifted. */
static CtRouteAccess const CtRouteAccessShift = 26;
/** A mask to select the route access flags. */
static CtRouteAccess const CtRouteAccessMask = 63U << CtRouteAccessShift;
/** A flag indicating that this route is one-way but is stored in the opposite direction to the allowed direction of travel. */
static CtRouteAccess const CtRouteAccessWrongWayFlag = 1 << 26;
/** A flag to forbid pedestrian access in a feature info object, or, in a vehicle type, to indicate a pedestrian. */
static CtRouteAccess const CtRouteAccessPedestrianFlag = 1 << 27;
/** A flag to forbid cycle access in a feature info object, or, in a vehicle type, to indicate a cyclist. */
static CtRouteAccess const CtRouteAccessCycleFlag = 1 << 28;
/** A flag to forbid motor vehicle access in a feature info object, or, in a vehicle type, to indicate a motor vehicle. */
static CtRouteAccess const CtRouteAccessMotorVehicleFlag = 1 << 29;
/** A flag to forbid emergency vehicle access in a feature info object, or, in a vehicle type, to indicate an emergency vehicle. */
static CtRouteAccess const CtRouteAccessEmergencyVehicleFlag = 1 << 30;
/** A flag to indicate the presence of other access restrictions in a feature info object, or, in a vehicle type, to indicate a type of vehicle not covered by the other flags. */
static CtRouteAccess const CtRouteAccessOtherFlag = 1U << 31;
/** A set of flags covering all vehicles. */
static CtRouteAccess const CtRouteAccessVehicle = CtRouteAccessCycleFlag | CtRouteAccessMotorVehicleFlag | CtRouteAccessEmergencyVehicleFlag;
/** A set of flags indicating normal access: that is, access for pedestrians, cyclists and motor vehicles. */
static CtRouteAccess const CtRouteAccessNormal = CtRouteAccessCycleFlag | CtRouteAccessMotorVehicleFlag | CtRouteAccessPedestrianFlag;

/** A map object: a point, line, polygon or array (texture), with its name, layer, ID and other attributes. */
@interface CtMapObject: NSObject <CtPath>
    /** Initializes a CtMapObject with a C++ map object; for internal use only. */
    -(id)initWithMapObject:(void*)aMapObject;
    /** Destroys the map object. */
    -(void)dealloc;
    /** The type of the map object: point, line, polygon or array. */
    @property (nonatomic) CtMapObjectType type;
    /** The name of the map layer to which the map object belongs. */
    @property (nonatomic) NSString* layerName;
    /** The label or standard name of the map object. */
    @property (nonatomic) NSString* label;
    /** Gets any string attribute. If aAttribName is null, gets the label. */
    -(NSString*)stringAttribute:(NSString*)aAttribName;
    /** Gets all the string attributes as a dictionary of key-value pairs. */
    -(NSMutableDictionary*)stringAttributes;
    /** The feature info of a map object. */
    @property (nonatomic) CtFeatureInfo* featureInfo;
    /**
    The center of a map object in map coordinates.
    The center is defined as the center of gravity for polygons and arrays,
    a point half way along the path for lines, and the point itself for points.
    */
    @property (nonatomic) CtPoint center;
    
    /** The axis-aligned bounding box of a map object, in map coordinates. */
    @property (nonatomic) CtRect bounds;

    /**
    The center of a map object in degrees.
    The center is defined as the center of gravity for polygons and arrays,
    a point half way along the path for lines, and the point itself for points.
    Because the center is calculated using map coordinates then converted to degrees,
    it depends on the map projection used by the map object.
    */
    @property (nonatomic) CtPointAndResult centerInDegrees;

    /** The bounding box of a map object in degrees. */
    @property (nonatomic) CtRectAndResult boundsInDegrees;
    
    /** The geometry of a map object in degrees. */
    @property (nonatomic) CtGeometryAndResult* geometryInDegrees;
    /**
    Finds the first string attribute matching aText, using aMatchMethod, and returns
    information about it.
    If aAttributes is non-null this function examines only attributes in the supplied comma-separated list,
    otherwise it examines all attributes except those starting with an underscore, which by convention are non-textual.
    Attribute names may contain the wild cards ? and *.
    If aPhrase is true matches of phrases within the attribute are allowed.
    */
    -(CtMatch*)match:(NSString*)aText matchMethod:(CtStringMatch)aMethod attributes:(NSString*)aAttributes phrase:(bool)aPhrase;
    /** The object ID of the map object, which can be used to delete objects inserted at run-time. */
    @property (nonatomic) uint64_t objectId;
    /** Gets the number of contours (sub-paths) in the path. */
    -(int)contourCount;
    /** Gets the number of points in a certain contour. */
    -(int)pointCount:(int)aContourIndex;
    /** Gets a point, given its contour index and point index. Returns (0,0) if the contour index or point index is out of range. */
    -(CtPathPoint)point:(int)aContourIndex pointIndex:(int)aPointIndex;
    /** The area of the map object in square meters. Returns zero for a point object. */
    @property (nonatomic) double area;
    /** The length or perimeter of the map object in meters. Returns zero for a point object. */
    @property (nonatomic) double lengthOrPerimeter;

    /** Gets a pointer to the C++ map object; for internal use only. */
    -(void*)object;
@end
/** A synonym of CtMapObject. */
@compatibility_alias CartoTypeMapObject CtMapObject;

/** A group of map objects with the same name, returned by a search function. */
@interface CtMapObjectGroup: NSObject
    /** Initializes a CtMapObjectGroup with a C++ CMapObjectGroup; for internal use only. */
    -(id)initWithMapObjectGroup:(void*)aMapObjectGroup;
    /** The name, summary address or other common attribute of all the objects. */
    @property (nonatomic, strong) NSString* name;
    /** The map objects. */
    @property (nonatomic, strong) NSMutableArray* objectArray;
@end
/** A synonym of CtMapObjectGroup. */
@compatibility_alias CartoTypeMapObjectGroup CtMapObjectGroup;

/** The ways two paths can intersect. */
typedef NS_ENUM(NSInteger,CtPathIntersectionType)
    {
    /** The intersection type is unknown. */
    CtPathIntersectionTypeUnknown,
    /** The paths do not intersect. */
    CtPathIntersectionTypeSeparate,
    /** The paths intersect or are identical. */
    CtPathIntersectionTypeIntersects,
    /** The first path contains the second. */
    CtPathIntersectionTypeContains,
    /** The second path contains the first. */
    CtPathIntersectionTypeContained
    };
/** A synonym of CtPathIntersectionType. */
typedef CtPathIntersectionType CartoTypePathIntersectionType;

/** Information about the intersection of two paths and their distance apart. */
@interface CtPathIntersectionInfo: NSObject
    /** The intersection type. */
    @property (nonatomic) CtPathIntersectionType type;
    /**
    The distance between the paths. The units are meters for functions involving
    map objects, otherwise they are the units used by the paths.
    */
    @property (nonatomic) double distance;
    /**
    The nearest point on the first path. The units are latitude and longitude for functions involving
    map objects, otherwise they are the units used by the paths.
    */
    @property (nonatomic) CtPoint nearest1;
    /**
    The nearest point on the second path. The units are latitude and longitude for functions involving
    map objects, otherwise they are the units used by the paths.
    */
    @property (nonatomic) CtPoint nearest2;

    /** For internal use: gets a pointer to the TPathIntersectionInfo object. */
    -(void*)pointer;
@end
/** A synonym of CtPathIntersectionInfo. */
@compatibility_alias CartoTypePathIntersectionInfo CtPathIntersectionInfo;

/** Types of clipping done by the Clip functions of the Geometry class. */
typedef NS_ENUM(NSInteger,CtClipOperation)
    {
    /** Returns the intersection of two paths; commutative. */
    CtClipOperationIntersection,
    /** Returns the union of two paths; commutative. */
    CtClipOperationUnion,
    /** Returns the difference of two paths; non-commutative. */
    CtClipOperationDifference,
    /** Returns the exclusive-or of the two paths; that is, any regions which are in neither path; commutative. */
    CtClipOperationXor
    };
/** A synonym of CtClipOperation. */
typedef CtClipOperation CartoTypeClipOperation;

/** Geometry: a point, polyline or poly-polygon with a coordinate type. */
@interface CtGeometry: NSObject
    /** Initializes the geometry object with the coordinate type to be used when adding points. */
    -(id)initWithCoordType:(CtCoordType)aCoordType;
    /** Creates a geometry object from a map object. */
    -(id)initWithMapObject:(CtMapObject*)aMapObject;
    /** Creates a geometry object from a single contour of another geometry object. */
    -(id)initWithGeometryContour:(CtGeometry*)aGeometry contourIndex:(int)aContourIndex;
    /** Creates a geometry object from a rectangle. */
    -(id)initWithRect:(CtRect)aRect coordType:(CtCoordType)aCoordType;
    /** Destroys the geometry object. */
    -(void)dealloc;
    /** Returns the coordinate type. */
    -(CtCoordType)coordType;
    /** Adds a new point. */
    -(void)appendX:(double)aX andY:(double)aY;
    /** Adds a new point, specifying the point type as well as the coordinates. */
    -(void)appendX:(double)aX andY:(double)aY andPointType:(CtPointType)aPointType;
    /** Starts a new contour. */
    -(void)beginContour;
    /** Returns the number of contours. */
    -(int)contourCount;
    /** Returns true if a contour is an outer contour, defined as being anti-clockwise, that is, having a positive area when treating all points as on-curve. Useful only for closed geometries. */
    -(bool)contourIsOuter:(int)aContourIndex;
    /** Returns the number of points in a contour. */
    -(int)pointCount:(int)aContourIndex;
    /** Returns a point, given a contour index and a point index. */
    -(CtPathPointFP)point:(int)aContourIndex pointIndex:(int)aPointIndex;
    /** Returns true if the geometry has no points. */
    -(bool)isEmpty;
    /** Returns true if the geometry is made from closed paths. */
    -(bool)isClosed;
    /** Sets the closed state. */
    -(void)setClosed:(bool)aClosed;
    /** Reverses the order of the points in each contour. */
    -(void)reverse;
    /** Interpolates points at a maximum distance apart to convert lines to great circles. Works only with degree coordinates, and with on-curve points. */
    -(CtResult)convertLinesToGreatCircles:(double)aMaxDistanceInMeters;
    /** Returns the intersection type of this geometry with a map object. */
    -(CtPathIntersectionType)intersectionType:(CtMapObject*)aMapObject;
    /**
    Returns information about the intersection of this geometry with a map object.
    The nearest positions are in degrees (X = longitude, Y = latitude) and the the distance is in meters.
    */
    -(CtPathIntersectionInfo*)intersectionInfo:(CtMapObject*)aMapObject;
    /** Returns the intersection types of this geometry with an array of map objects.  The types are returned as integers because of the difficulty of returning an array of enumeration values in Objective C. */
    -(NSArray<NSNumber*>*)intersectionTypes:(NSArray<CtMapObjectGroup*>*)aMapObjectArray;
    /**
    Returns information about the intersection of this geometry with a sequence of map objects returned by a function.
    The nearest positions are in degrees (X = longitude, Y = latitude) and the the distances are in meters.
    */
    -(NSArray<CtPathIntersectionInfo*>*)intersectionInfos:(NSArray<CtMapObjectGroup*>*)aMapObjectArray;
    /** Returns true if this geometry intersects, contains or is contained by the map object aMapObject. */
    -(bool) intersects:(CtMapObject*)aMapObject;
    /** Returns true if this geometry contains the map object aMapObject. */
    -(bool) contains:(CtMapObject*)aMapObject;
    /** Returns true if this geometry is contained by the map object aMapObject. */
    -(bool) contained:(CtMapObject*)aMapObject;
    /**
    Returns the result of clipping a map object using this geometry, which must be closed.
    Supports intersection, union, difference and exclusive-or operations.
    All operations except difference are commutative.
    The difference operation subtracts this geometry from aMapObject.
    
    The returned geometry object is in degrees.
    */
    -(CtGeometry*) clip:(CtClipOperation)aClipOperation mapObject:(CtMapObject*)aMapObject;
    /**
    Returns the results of clipping an array of map objects using this geometry, which must be closed.
    Supports intersection, union, difference and exclusive-or operations.
    All operations except difference are commutative.
    The difference operation subtracts this geometry from the map objects.
    
    The returned geometry objects are in degrees.
    */
    -(NSArray<CtGeometry*>*) clip:(CtClipOperation)aClipOperation mapObjects:(NSArray<CtMapObjectGroup*>*)aMapObjectArray;
    /**
    Returns a geometry that is the envelope of a map object,
    created by offsetting the path by aOffset. The offset
    may be negative if the subject path is made of polygons only.
    
    The returned geometry object is in degrees.
    */
    +(CtGeometry*) envelope:(CtMapObject*)aMapObject offsetInMeters:(double)aOffsetInMeters;

    /** (for internal use) Initializes the geometry object with an underlying C++ geometry object. */
    -(id)initWithGeometry:(void*)aGeometry;
    /** (for internal use) Gets a pointer to the underlying C++ geometry object. */
    -(void*)geometry;
@end
/** A synonym of CtGeometry. */
@compatibility_alias CartoTypeGeometry CtGeometry;

/** Parameters for the general find function. */
@interface CtFindParam: NSObject
    /** The default initializer. */
    -(id)init;
    /** Creates find parameters for finding nearby. */
    -(id)initWithFeatureType:(CtFeatureType)aFeatureType text:(NSString*)aText location:(CtGeometry*)aLocation;

    /** The maximum number of objects to return. */
    @property (nonatomic) int maxObjectCount;
    /** The clip path; no clipping is done if clip is empty. */
    @property (nonatomic) CtGeometry* clip;
    /** The current location. If it is non-empty, objects in or near this region are preferred. */
    @property (nonatomic) CtGeometry* location;
    /** If non-null, a list separated by spaces or commas. If it is null all layers are searched. */
    @property (nonatomic, strong) NSString* layers;
    /**
    If 'attributes' is null or empty, search all the attributes, otherwise it is a list of
    attributes separated by spaces or commas; use '$' to indicate the label (the unnamed attribute).
    */
    @property (nonatomic, strong) NSString* attributes;
    /**
    If non-null, restricts the search to objects containing a string
    in one of their string attributes.
    */
    @property (nonatomic, strong) NSString* text;
    /** The string matching method used for text searching. */
    @property (nonatomic) CtStringMatch stringMatchMethod;
    /**
    If non-null, a condition expression (e.g., "Type==2")
    which must be fulfilled by all the objects.
    */
    @property (nonatomic, strong) NSString* condition;
    /** If merge is true adjoining objects with the same name and attributes may be merged into a single object. The default value is false. */
    @property (nonatomic) bool merge;
    /**
    The maximum time in seconds allowed for a find operation. Find operations are not guaranteed to
    return in this time or less, but will attempt to do so. A timeout of zero or less means there is no timeout.
    */
    @property (nonatomic) double timeOut;
@end
/** A synonym of CtFindParam. */
@compatibility_alias CartoTypeFindParam CtFindParam;

/** Types of writable map. */
typedef NS_ENUM(NSInteger,CtWritableMapType)
    {
    /** A writable map stored in memory. */
    CtWritableMapTypeMemory,
    /** A writable map stored in an SQLite database. */
    CtWritableMapTypeSQLite
    };
/** A synonym of CtWritableMapType. */
typedef CtWritableMapType CartoTypeWritableMapType;

/** Anchor positions for notices like the legend or scale bar. */
typedef NS_ENUM(NSInteger,CtNoticeAnchor)
    {
    /** The top left corner of the display. */
    CtNoticeAnchorTopLeft,
    /** The top right corner of the display. */
    CtNoticeAnchorTopRight,
    /** The bottom left corner of the display. */
    CtNoticeAnchorBottomLeft,
    /** The bottom right corner of the display. */
    CtNoticeAnchorBottomRight,
    /** The center of the top edge of the display. */
    CtNoticeAnchorTop,
    /** The center of the right side of the display. */
    CtNoticeAnchorRight,
    /** The center of the left side of the display. */
    CtNoticeAnchorLeft,
    /** The center of the bottom edge of the display. */
    CtNoticeAnchorBottom,
    /** The center of the display. */
    CtNoticeAnchorCenter
    };
/** A synonym of CtNoticeAnchor. */
typedef CtNoticeAnchor CartoTypeNoticeAnchor;

/**
Positions for notices like the legend or scale bar.
 
Notices are positioned by moving them inwards from the designated position
by the specified insets. For central positions (e.g., the X inset in Top,
or both insets in Center) X insets move the position right and Y insets move it down.
Units may be 'cm', 'mm', 'in', 'pt' (points: 1/72in), or 'pc' (picas: 1/6in).
 
The default inset is 3mm, or zero for central positions.
*/
@interface CtNoticePosition: NSObject
    /** Creates a notice position from an anchor position. */
    -(id)init:(CtNoticeAnchor)aAnchor;
    /** Creates a notice position from an achor position,width and X and Y insets. Any values with null or empty units are ignored, and default values are used. */
    -(id)init:(CtNoticeAnchor)aAnchor width:(double)aWidth widthUnit:(NSString*)aWidthUnit xInset:(double)aXInset xUnit:(NSString*)aXUnit yInset:(double)aYInset yUnit:(NSString*)aYUnit;
    /** Returns the native position object. For internal use only. */
    -(void*)position;
@end
/** A synonym of CtNoticePosition. */
@compatibility_alias CartoTypeNoticePosition CtNoticePosition;

/** A bitmap and a position to draw it. Used when drawing notices on the map. */
@interface CtPositionedBitmap: NSObject
    /** Creates a positioned bitmap from a bitmap and a position. */
    -(id)init:(UIImage*)aImage topLeft:(CGPoint)aTopLeft;
    /** Destroys the object. */
    -(void)dealloc;
    /** The bitmap. */
    @property (nonatomic, strong) UIImage* image;
    /** The position at which to draw the bitmap on the map. */
    @property (nonatomic) CGPoint topLeft;
@end
/** A synonym of CtPositionedBitmap. */
@compatibility_alias CartoTypePositionedBitmap CtPositionedBitmap;

/** Parameters giving detailed control of the perspective view. */
@interface CtPerspectiveParam: NSObject
    /** Reads a perspective parameter object from its XML form. */
    -(CtResult)readFromXml:(NSString*)aText;
    /** Creates a string representing a perspective parameter object in XML. */
    -(NSString*)toXml;
    /** The position of the point on the terrain below the camera, in degrees longitude (x) and latitude (y). */
    @property (nonatomic) CtPoint positionDegrees;
    /** If true, ignore iPosition, and set the camera position so that the location in the center of the display is shown. */
    @property (nonatomic) bool autoPosition;
    /** The height of the camera above the terrain. The value 0 causes a default value to be used, which is the width of the display. */
    @property (nonatomic) double heightMeters;
    /** The azimuth of the camera in degrees going clockwise, where 0 is N, 90 is E, etc. */
    @property (nonatomic) double azimuthDegrees;
    /** If true, ignore iAzimuthDegrees and use the current map orientation. */
    @property (nonatomic) bool autoAzimuth;
    /** The declination of the camera downward from the horizontal plane. Values are clamped to the range -90...90. */
    @property (nonatomic) double declinationDegrees;
    /** The amount by which the camera is rotated about its axis, after applying the declination, in degrees going clockwise. */
    @property (nonatomic) double rotationDegrees;
    /** The camera's field of view in degrees. */
    @property (nonatomic) double fieldOfViewDegrees;
    /** (for internal use) Gets a pointer to the underlying C++ object. */
    -(void*)getPerspectiveParam;
@end
/** A synonym of CtPerspectiveParam. */
@compatibility_alias CartoTypePerspectiveParam CtPerspectiveParam;

/** The view state, which can be retrieved or set so that maps can be recreated. */
@interface CtViewState: NSObject
    /** Reads a view state from its XML form. */
    -(CtResult)readFromXml:(NSString*)aText;
    /** Creates a string representing a view state in XML. */
    -(NSString*)toXml;
    /** The display width in pixels. */
    @property (nonatomic) int widthInPixels;
    /** The display height in pixels. */
    @property (nonatomic) int heightInPixels;
    /** The view center in map coordinates. */
    @property (nonatomic) CtPoint viewCenterDegrees;
    /** The denominator of the scale fraction; e.g., 50000 for 1:50000. */
    @property (nonatomic) double scaleDenominator;
    /** The clockwise rotation of the view in degrees. */
    @property (nonatomic) double rotationDegrees;
    /** True if perspective mode is on. */
    @property (nonatomic) bool perspective;
    /** Gets a copy of the parameters to be used for perspective mode. */
    -(CtPerspectiveParam*)getPerspectiveParam;
    /** Sets the parameters to be used for perspective mode. */
    -(void)setPerspectiveParam:(CtPerspectiveParam*)aPerspectiveParam;
    /** (for internal use) Gets a pointer to the underlying C++ object. */
    -(void*)viewState;
@end
/** A synonym of CtViewState. */
@compatibility_alias CartoTypeViewState CtViewState;

/** The case of letters and strings in Latin, Greek, Cyrillic and other alphabetic scripts. */
typedef NS_ENUM(NSInteger,CtLetterCase)
    {
    /** A code used when the letter case is to be left as it is, or is irrelevant. */
    CtLetterCaseNone,
    /** The case of the letters 'a', 'b', c', etc. */
    CtLetterCaseLower,
    /** The case of the letters 'A', 'B', 'C, etc. */
    CtLetterCaseUpper,
    /**
    The case of letters like U+01C8 'Lj' and a few others; also indicates
    that the first letter of each word should be changed to titlecase.
    */
    CtLetterCaseTitle
    };
/** A synonym of CtLetterCase. */
typedef CtLetterCase CartoTypeLetterCase;
    
/** Identifiers of tables in CtM1 map files. */
typedef uint16_t CtMapTableId NS_TYPED_ENUM;

/** The ID of the global information table. */
static CtMapTableId const CtMapTableIdGlobal = 0;
/** The ID of the table containing the layers containing the map objects; see also KLowResolutionLayerTable. */
static CtMapTableId const CtMapTableIdLayer = 1;
/** The ID of the the table containing the map projection. */
static CtMapTableId const CtMapTableIdProjection = 4;
/** The ID of the table of compressed strings referenced by the map objects. */
static CtMapTableId const CtMapTableIdString = 5;
/** The ID of the table containing layer data containing map objects at lower resolutions, simplified for display at smaller scales; see also KLayerTable. */
static CtMapTableId const CtMapTableIdLowResolutionLayer = 6;
/** The ID of the table containing color palettes for raster image objects. */
static CtMapTableId const CtMapTableIdPalette = 7;
/** The ID of the table containing the text index used when searching for string attributes. */
static CtMapTableId const CtMapTableIdTextIndex = 8;
/** The ID of the A-star routing data table. */
static CtMapTableId const CtMapTableIdRouteAStar = 10;
/** The ID of the contraction hierarchy routing data table. */
static CtMapTableId const CtMapTableIdRouteCH = 11;
/** The ID of the table containing extra information used for A-star routing. */
static CtMapTableId const CtMapTableIdRouteExtra = 12;
/** The ID of the turn-expanded routing data table. */
static CtMapTableId const CtMapTableIdRouteTurnExpanded = 13;
/** The ID of the table containing contraction hierarchy routing data that can optionally be used stand-alone, in combination with other map files. */
static CtMapTableId const CtMapTableIdRouteCHStandAlone = 14;
/** The ID of the turn-expanded contraction hierarchy routing data table. */
static CtMapTableId const CtMapTableIdRouteTECH = 15;
/** The ID of the table containing contraction hierarchy routing data that is a tile that can be used with other tiles from the same data set. */
static CtMapTableId const CtMapTableIdRouteCHTiled = 16;
/** The ID of the table containing turn-expanded contraction hierarchy routing data that is a tile that can be used with other tiles from the same data set. */
static CtMapTableId const CtMapTableIdRouteTECHTiled = 17;
/** The ID of the table containing compact turn-expanded routing data, which uses less run-time RAM. */
static CtMapTableId const CtMapTableIdRouteTurnExpandedCompact = 18;
/** An ID used when no table exists or the table type is unknown. */
static CtMapTableId const CtMapTableIdNone = UINT16_MAX;

/**
The format used for points in CTM1 data.
These numbers must fit into 8 bits for the moment because of the way they are stored in the CTM1 file.
The values are discontinuous because several obsolete values have been discarded.
*/
typedef NS_ENUM(NSInteger,CtPointFormat)
    {
    /** An arbitrary or unknown point format. */
    CtPointFormatUnknown = 0,

    /** Units are projected map meters. */
    CtPointFormatMeter = 2,

    /**
    Units are degrees as 11.21 fixed-point numbers.
    That is, there are 2^21 (2097152) units to a degree.
    This format is used only by the makemap tool, for representing
    unprojected map data.
    */
    CtPointFormatScaledDegree = 7,

    /**
    Units are 32nds of projected map meters.
    This is the default format for CTM1 data.
    */
    CtPointFormatMeter32nds = 8
    };
/** A synonym of CtPointFormat. */
typedef CtPointFormat CartoTypePointFormat;

/** Metadata describing a CTM1 map file. */
@interface CtMapMetaData: NSObject
    /** The major part of the CTM1 format version. */
    @property (nonatomic) int fileVersionMajor;
    /** The minor part of the CTM1 format version. */
    @property (nonatomic) int fileVersionMinor;
    /** The major part of the version of CartoType used to build the makemap tool which created the CTM1 file. */
    @property (nonatomic) int CartoTypeVersionMajor;
    /** The minor part of the version of CartoType used to build the makemap tool which created the CTM1 file. */
    @property (nonatomic) int CartoTypeVersionMinor;
    /** The CartoType build (version control revision number) used to build the makemap tool which created the CTM1 file. */
    @property (nonatomic) int CartoTypeBuild;
    /** The name of the map data set. */
    @property (nonatomic) NSString* dataSetName;
    /** The copyright notice applying to the map data. */
    @property (nonatomic) NSString* copyright;
    /** The name of the map projection. */
    @property (nonatomic) NSString* projectionName;
    /** The layers. */
    @property (nonatomic) NSMutableArray<NSString*>* layers;
    /** The Proj4 parameters for the map projection. */
    @property (nonatomic) NSString* projectionParameters;
    /** The point format for map coordinates: either Meter or Meter32nds. */
    @property (nonatomic) CtPointFormat pointFormat;
    /** The axis-aligned bounds of the map, in map coordinates. */
    @property (nonatomic) CtRect extentInMapCoords;
    /** The axis-aligned bounds of the map, in degrees of longitude and latitude. */
    @property (nonatomic) CtRect extentInDegrees;
    /** The route table type:. */
    @property (nonatomic) CtMapTableId routeTableType;
    /** True if the route data contains gradients. */
    @property (nonatomic) bool routeDataHasGradients;
    /** True if file positions in the data file take up 5 bytes rather than 4. */
    @property (nonatomic) bool largeFile;
    /** True if the driving side (rule of the road) is known. */
    @property (nonatomic) bool drivingSideKnown;
    /** True if the driving side is known and the rule is to drive on the left. */
    @property (nonatomic) bool driveOnLeft;
    
    /** Initializes with a C++ metadata object: for internal use only. */
    -(id)initWithMetaData:(void*)aMetaData;
    /** Returns a pointer to the C++ metadata object: for internal use only. **/
    -(const void*)metaData;
@end
/** A synonym of CtMapMetaData. */
@compatibility_alias CartoTypeMapMetaData CtMapMetaData;

/**
Parameters used when inserting a map object into a writable map.
After creating the parameter object, add points and optionally set other parameters,
then call one of the functions which insert objects.
*/
@interface CtMapObjectParam: NSObject
    /** Initializes the parameter object with an object type, layer name and the coordinate type to be used when adding points. */
    -(id)initWithType:(CtMapObjectType)aType andLayer:(NSString*)aLayer andCoordType:(CtCoordType)aCoordType;
    /** Destroys the parameter object. */
    -(void)dealloc;
    /** Adds a new point to the object's geometry. */
    -(void)appendX:(double)aX andY:(double)aY;
    /** Adds a new point to the object's geometry, specifying the point type as well as the coordinates. */
    -(void)appendX:(double)aX andY:(double)aY andPointType:(CtPointType)aPointType;
    /** Starts a new contour; use this if the object is made of multiple contours. */
    -(void)beginContour;
    /** (for internal use) Gets a pointer to the underlying C++ geometry object. */
    -(void*)geometry;

    /** The map handle: 0 can be used for the writable in-memory map. */
    @property (nonatomic) uint32_t mapHandle;
    /** The name of the layer. */
    @property (nonatomic, strong) NSString* layer;
    /** The radius (default = 0); if the radius is greater than zero, a point object is turned into a circle with the given radius, and other objects are made into envelopes of that size. */
    @property (nonatomic) double radius;
    /** The X radius (default = 0); if x and y radii are greater than zero, a point object is turned into an ellipse with the given radii. */
    @property (nonatomic) double radiusX;
    /** The Y radius (default = 0); if x and y radii are greater than zero, a point object is turned into an ellipse with the given radii. */
    @property (nonatomic) double radiusY;
    /** The coordinate type of the radii (default = MapMeterCoordType). */
    @property (nonatomic) CtCoordType radiusCoordType;
    /** The rotation in degrees applied to an ellipse map object. */
    @property (nonatomic) double rotationDegrees;
    /** Integer attribute value (default = 0). */
    @property (nonatomic, strong) CtFeatureInfo* featureInfo;
    /** String attributes if any (default = null); if there are no string attributes this value map be null. */
    @property (nonatomic, strong) NSString* stringAttrib;
    /** The object identifier (default = 0); if it is zero an identifier is assigned and returned in this property */
    @property (nonatomic) uint64_t objectId;
    /** If true (the default) replace the object if its ID is a duplicate of that of an existing object. */
    @property (nonatomic) bool replace;
@end
/** A synonym of CtMapObjectParam. */
@compatibility_alias CartoTypeMapObjectParam CtMapObjectParam;

/** Parameters specifying map files (CTM1 files) representing tiles to be loaded and unloaded automatically. */
@interface CtTileParam: NSObject
    /**
    Adds a tile set to the tile parameters

    @param aFileNameTemplate A C++ std::format template used to create each filename, using integer arguments zoom, x, y in that order.
    For example, "/maps/tile-zoom{}-x{}-y{}.ctm1".
    The arguments may be reordered using argument indexes: see the C++ std::format documentation for details.

    @param aTileRange The inclusive range of tiles that is available.

    @param aFileZoom The zoom level of these tiles.

    @param aMinDisplayZoom The minimum zoom level at which these tiles are to be drawn.

    @param aMaxDisplayZoom The maximum zoom level at which these tiles are to be drawn.
    */
    -(void)addTileSet:(NSString*)aFileNameTemplate tileRange:(CtRect)aTileRange fileZoom:(int)aFileZoom minDisplayZoom:(int)aMinDisplayZoom maxDisplayZoom:(int)aMaxDisplayZoom;

    /** Returns a pointer to the C++ TileParam object: for internal use only. **/
    -(const void*)tileParam;
@end
/** A synonym of CtTileParam. */
@compatibility_alias CartoTypeTileParam CtTileParam;


#endif
