/*
CartoTypeLegend.h
Copyright 2015-2024 CartoType Ltd.
See www.cartotype.com for more information.
*/

#ifndef CartoTypeLegend_h__
#define CartoTypeLegend_h__

#import <CartoType/CartoTypeBase.h>

/**
\file
The CtLegend class, which is used for map legends, turn instructions, scale bars and copyright notices.
*/

/** Style flags used in the Legend constructor. */
typedef NS_OPTIONS(uint32_t,CtLegendStyle)
    {
    /** A style flag used in the Legend constructor to add the name of the main map as a title. */
    CtLegendStyleFlagTitle = 1,
    /** A style flag used in the Legend constructor to add the scale in the form 1:NNN to the title. */
    CtLegendStyleFlagScaleInTitle = 2,
    /** A style flag used in the Legend constructor to add lines for common map objects including roads, forests, parks and stations. */
    CtLegendStyleFlagMapObjects = 4,
    /** A style flag used in the Legend constructor to add a scale bar. */
    CtLegendStyleFlagScaleBar = 8,
    /** Style flags used in the Legend constructor to select the standard style. */
    CtLegendStyleStandard = CtLegendStyleFlagTitle | CtLegendStyleFlagScaleInTitle | CtLegendStyleFlagMapObjects | CtLegendStyleFlagScaleBar,
    /**
    Style flags used in the constructor to create an empty legend object suitable for turn instructions.
    The font, border, colors, etc., of the legend can then be changed, after which the legend can be used in SetTurnInstructions.
    */
    CtLegendStyleTurn = 16,
    /**
    Style flags used in the constructor to create an empty legend object suitable for a scale bar.
    The font, border, colors, etc., of the legend can then be changed, after which the legend can be used in SetScaleBar.
    */
    CtLegendStyleScale = 32,
    /** Style flags used in the Legend constructor to create an empty legend object. */
    CtLegendStyleEmpty = 0
    };
/** A synonym of CtLegendStyle. */
typedef CtLegendStyle CartoTypeLegendStyle;

@class CtFramework; // forward declaration

/**
The CtLegend class specifies legends or map keys showing sample map objects, with optional text lines and a scale.
 
Dimensions are specified with units, which may be "pt" (point), "pc" (pica), "cm" (centimetre),
"mm" (millimetre), "in" (inch), or "px" (pixel). An empty or unrecognised unit is taken as "px".
 
To use a legend on a map, create a Legend object and pass it to the setLegend method of CtFramework.
*/
@interface CtLegend: NSObject

/** Initializes a CtLegend with a CtFramework object, using the same fonts and style sheet as aFramework. */
-(id)initWithFramework:(CtFramework*)aFramework;

/** Initializes a CtLegend with a CtFramework object, using the same fonts and style sheet as aFramework, and a style made from the CtLegendStyle constants. */
-(id)initWithFramework:(CtFramework*)aFramework style:(CtLegendStyle)aStyle;

/** Destroys the object. */
-(void)dealloc;

/** Deletes all legend lines (map objects, text lines and scale bars). */
-(void)clear;

/**
Adds a line containing a map object, with an optional label to the left of it.
If the parameter aOsmType is a type code made from three lower-case letters in the range a...z, it is converted to an integer type code which is ORed with aIntAttrib.
*/
-(void)addMapObjectLine:(CtMapObjectType)aType layer:(NSString*)aLayer freatureInfo:(CtFeatureInfo*)aFeatureInfo stringAttrib:(NSString*)aStringAttrib label:(NSString*)aLabel;

/** Adds a line of text, using the current font family, font size, text color and minimum line height. */
-(void)addTextLine:(NSString*)aText;

/**
Adds a scale bar line, using the current font family, font size, alignment, text color and minimum line height.
*/
-(void)addScaleLine;

/** Sets the main style sheet, replacing the style sheet copied from the Framework parameter to the Legend constructor. */
-(void)setMainStyleSheet:(NSString*)aData;

/**
Sets the extra style sheet.
 
The purpose of this function is to allow a caller such as a style sheet editor
to create a style sheet containing a single layer, or a small number of
layers, and draw a legend for that layer only.
*/
-(void)setExtraStyleSheet:(NSString*)aData;

/** Sets the background color. Transparent colors are allowed. */
-(void)setBackgroundColor:(CtColor)aColor;

/** Sets the border color and corner radius for the whole legend. Transparent border colors are allowed. */
-(void)setBorderColor:(CtColor)aColor strokeWidth:(double)aStrokeWidth radius:(double)aRadius unit:(NSString*)aUnit;

/** Sets the margin width for the whole legend. */
-(void)setMarginWidth:(double)aMarginWidth unit:(NSString*)aUnit;

/** Sets the minimum line height for subsequently added lines. */
-(void)setMinLineHeight:(double)aLineHeight unit:(NSString*)aUnit;

/** Sets the label wrap width for subsequently added lines. Values of zero or less prevent wrapping. */
-(void)setLabelWrapWidth:(double)aWrapWidth unit:(NSString*)aUnit;

/** Sets the font family for subsequently added lines. */
-(void)setFontFamily:(NSString*)aFontFamily;

/** Sets the font style for subsequently added lines. The style is made from the CtFontStyle constants. */
-(void)setFontStyle:(uint32_t)aStyle;

/** Sets the font size for subsequently added lines. */
-(void)setFontSize:(double)aFontSize unit:(NSString*)aUnit;

/** Sets the text color for subsequently added lines. */
-(void)setTextColor:(CtColor)aTextColor;

/** Sets the diagram color, used for scale bars and turn diagrams, for subsequently added lines. */
-(void)setDiagramColor:(CtColor)aDiagramColor;

/** Sets the alignment for labels and scale bars. */
-(void)setAlignment:(CtAlign)aAlignment;

/** Sets the amount in degrees by which rectangles drawn for polygon legend lines are rotated, for subsequently added lines. */
-(void)setPolygonRotation:(double)aDegrees;

/** Returns true if any of the lines of this legend object is a turn instruction. */
-(bool)hasTurnInstruction;

/**
Sets the instruction to be displayed for the current turn, overriding automatically generated instructions.
To revert to automatically generated instructions, call this function with empty text.
*/
-(void)setTurnInstruction:(NSString*)aText;

/**
Returns the instruction to be displayed for the current turn, overriding automatically generated instructions.
Returns the empty string if automatically generated instructions are used and there is no overriding instruction.
*/
-(NSString*)turnInstruction;

/** Gets a pointer to the C++ CLegend object: internal use only. */
-(void*)legend;

@end
/** A synonym of CtLegend. */
@compatibility_alias CartoTypeLegend CtLegend;

#endif
