/*
CartoTypeResultCode.h
Copyright 2011-2024 CartoType Ltd.
See www.cartotype.com for more information.
*/

#ifndef CartoTypeResultCode_h__
#define CartoTypeResultCode_h__

/**
\file
CartoType result codes and the result type.
*/

/**
CartoType error codes.
CartoType functions actually return integers typed as CtResult, which
may be compared with these values.
*/ 
typedef NS_ENUM(NSInteger,CtError)
    {
    /** No error, a successful result. */
    CtErrorNone = 0,
    
    /**
    CtErrorGeneral is used where an error has occurred but no other CartoType
    error is applicable. For example, this error code can be used
    when FreeType returns an error code for illegal TrueType hinting instructions.
    */
    CtErrorGeneral = 1,
    
    /** Use CtErrorNoMemory when an attempted memory allocation fails. */
    CtErrorNoMemory = 2,
    
    /**
    CtErrorEndOfData is returned by iterators and streams when no more data
    is available. It may be treated as an error or not depending on the
    circumstances.
    */
    CtErrorEndOfData = 3,
    
    /**
    CtErrorTextUnmodifiable is returned when an attempt is made to
    modify an unmodifiable string.
    */
    CtErrorTextUnmodifiable = 4,
    
    /**
    CtErrorTextTruncated is returned when text inserted into
    a string is too long for a fixed-size
    buffer.
    */
    CtErrorTextTruncated = 5,
    
    /**
    CtErrorNotFound is returned whenever a resource like a file
    is not found, or a search fails. 
    It may be treated as an error or not depending on the
    circumstances.
    */
    CtErrorNotFound = 6,
    
    /**
    The code CtErrorUnimplemented is used for functions that have
    not yet been implemented, or for circumstances or options
    within functions that remain unimplemented because they
    are deemed to be either rarely or never useful.
    */
    CtErrorUnimplemented = 7,
    
    /**
    If the graphics system detects an invalid outline, such as one
    that starts with a cubic control point, it returns
    CtErrorInvalidOutline.
    */
    CtErrorInvalidOutline = 8,
    
    /**
    The code CtErrorIo should be used for unexpected read or write
    errors on files or other data streams. This is for where you
    attempt to read data that should be there, but the OS says
    it's unavailable (e.g., someone has taken the CD out of the drive
    or the network is down). 
    */
    CtErrorIo = 9,
    
    /**
    CtErrorCorrupt is returned when illegal values are found when reading data
    from a file or other serialized form.
    */
    CtErrorCorrupt = 10,
    
    /**
    This error is returned by database accessors, typeface accessors, etc.,
    to indicate that this data format is unknown to a particular accessor.
    It is not really an error but tells the caller to try the next
    accessor in the list.
    */
    CtErrorUnknownDataFormat = 11,
    
    /**
    This error is returned by transformation inversion if the transformation 
    has no inverse.
    */
    CtErrorNoInverse = 12,
    
    /**
    This error is returned by the projection system if the projection 
    causes overflow.
    */
    CtErrorProjectionOverflow = 13,
    
    /**
    The code CtErrorCancel is returned to cancel an asynchronous
    operation that uses callbacks. The callback or virtual function
    returns this code. the caller must terminate further processing
    on receipt of any error, but this one indicates that there has been
    no actual error, but the handler has canceled the operation.
    */
    CtErrorCancel = 14,
    
    /**
    This error is returned when an invalid argument has been given to a function.
    */
    CtErrorInvalidArgument = 15,
    
    /**
    This error is returned by a data reader when it can not deal with the data version.
    */
    CtErrorUnknownVersion = 16,
    
    /**
    This error is returned by the base library when reading data or calculations result
    in overflow.
     */
    CtErrorOverflow = 17,
    
    // ABOLISHED: CtErrorOverlap = 18,
    
    /**
    The error code returned by line intersection algorithms when the lines are parallel.
    */
    CtErrorParallelLines = 19,
    
    // ABOLISHED: CtErrorPathLengthExceeded = 20,
    
    // ABOLISHED: CtErrorMaxTurnExceeded = 21,
    
    /**
    An attempt was made to draw a bitmap needing a color palette
    but no palette was provided.
    */
    CtErrorNoPalette = 22,
    
    /**
    An attempt was made to insert a duplicate object into a collection
    class that does not allow duplicates.
    */
    CtErrorDuplicate = 23,
    
    /**
    The projection for converting latitude and longitude to map coordinates
    is unknown or unavailable.
    */
    CtErrorNoProjection = 24,
    
    /**
    A palette is full and no new entries can be added.
    */
    CtErrorPaletteFull = 25,
    
    /**
    The dash array for drawing strokes is invalid.
    */
    CtErrorInvalidDashArray = 26,
    
    /**
    The central-path system failed to create a usable path.
    */
    CtErrorCentralPath = 27,
    
    /**
    A route was needed by the navigation system
    but no route was available.
    */
    CtErrorNoRoute = 28,
    
    /**
    There was an attempt to zoom beyond the legal zoom limits for a map.
    */
    CtErrorZoomLimitReached = 29,
    
    /**
    There was an attempt to project a map object that had already been projected.
    */
    CtErrorAlreadyProjected = 30,
    
    /**
    Conditions are too deeply nested in the style sheet.
    */
    CtErrorConditionsTooDeeplyNested = 31,
    
    /**
    There was an attempt to use a null font for drawing text.
    */
    CtErrorNullFont = 32,
    
    /**
    An attempt to read data from the internet failed.
    */
    CtErrorInternetIo = 33,
    
    /**
    Division by zero in an interpreted expression.
    */
    CtErrorDivideByZero = 34,
    
    /**
    A transform failed because an argument or result was out of range.
    */
    CtErrorTransformFailed = 35,
    
    /**
    Reading a bitmap from PNG format failed.
    */
    CtErrorPngRead = 36,
    
    /**
    Reading a bitmap from JPG format failed.
    */
    CtErrorJpgRead = 37,
    
    /**
    An object did not intersect a specified region.
    */
    CtErrorNoIntersection = 38,
    
    /**
    An operation was interrupted, for example by another thread writing to a shared flag.
    */
    CtErrorInterrupt = 39,
    
    /**
    There was an attempt to use map databases of incompatible point format,
    datum or axis orientations to draw a map or find objects in a map.
    */
    CtErrorMapDataBaseFormatMismatch = 40,
    
    /** A key supplied for encryption was too short. */
    CtErrorEncryptionKeyTooShort = 41,
    
    /** No encryption key has been set. */
    CtErrorNoEncryptionKey = 42,
    
    /** A code for standard emergency messages. */
    CtErrorEmergencyMessage = 43,
    
    /** A code for standard alert messages. */
    CtErrorAlertMessage = 44,
    
    /** A code for standard critical messages. */
    CtErrorCriticalMessage = 45,
    
    /** A code for standard error messages. */
    CtErrorErrorMessage = 46,
    
    /** A code for standard warning messages. */
    CtErrorWarningMessage = 47,
    
    /** A code for standard notice messages. */
    CtErrorNoticeMessage = 48,
    
    /** A code for standard information messages. */
    CtErrorInfoMessage = 49,
    
    /** A code for standard debug messages. */
    CtErrorDebugMessage = 50,
    
    /** A function has been called which is available only when navigating. */
    CtErrorNotNavigating = 51,
    
    /** The global framework object does not exist. */
    CtErrorNoFramework = 52,
    
    /** The global framework object already exists. */
    CtErrorFrameworkAlreadyExists = 53,
    
    /** A string was not transliterable. */
    CtErrorUntransliterable = 54,
    
    /** Writing a bitmap to PNG format failed. */
    CtErrorPngWrite = 55,
    
    /** There was an attempt to write to a read-only map database. */
    CtErrorReadOnlyMapDataBase = 56,
    
    /** There was an error in the PROJ.4 projection library error other than a projection overflow. */
    CtErrorProj4 = 57,

    /** No route could be created because there were no roads near the start point of a route section. */
    CtErrorNoRoadsNearStartOfRoute = 59,

    /** No route could be created because there were no roads near the end point of a route section. */
    CtErrorNoRoadsNearEndOfRoute = 60,

    /** No route could be created because the start and end point were not connected. */
    CtErrorNoRouteConnectivity = 61,
    
    /** An unsupported feature was requested from the XML parser. */
    CtErrorXmlFeatureNotSupported = 62,
    
    /** A map file was not found. */
    CtErrorMapNotFound = 63,

    /** A font file was not found. */
    CtErrorFontNotFound = 64,

    /** A style sheet was not found. */
    CtErrorStyleSheetNotFound = 65,
    
    /** An attempt was made to set a route attribute on a feature info object that is not a route. */
    CtErrorFeatureInfoIsNotRoute = 66,

    /** An attempt was made to set a non-route attribute on a feature info object that is a route. */
    CtErrorFeatureInfoIsRoute = 67,

    /** No route data is available. */
    CtErrorNoRouteData = 68,

    /** Too many fonts are used in a style sheet. */
    CtErrorTooManyFontsInStyleSheet = 69,

    /** Too many box styles are used in a style sheet. */
    CtErrorTooManyBoxStylesInStyleSheet = 70,

    /** No road or other routable segment was found near one or more route points when calculating a route. */
    CtErrorNoRoad = 71,

    /** An attempt was made to use a route which is outside the current map. */
    CtErrorRouteOutsideMap = 72,
    
    /** An attempt was made to load a map which is already loaded.  */
    CtErrorMapAlreadyLoaded = 73,

    /**
    The start of the range of errors in XML parsing or in the style sheet format, such as a syntactically incorrect dimension or color.
    The top byte is 0x10 for style sheet errors, or in the range 0x11 to 0x2A for Expat XML parser error codes.

    The low three bytes give the error location: one byte for the column number (clamped to 0...255) two bytes for the line number (clamped to 0...65535).
    */
    CtErrorXmlRangeStart = 0x10000000,

    /** The end of the range of errors in XML parsing or in the style sheet format. */
    CtErrorXmlRangeEnd = 0x2AFFFFFF,
    
    /**
    The base of error codes for returning SQLite errors.
    The SQLite error code is placed in the lower two bytes.
    */
    CtErrorSQLite = 0x30000000,
    
    /** The base for system errors. The system error number is placed in the lower 16 bits as a signed number. */
    CtErrorSystem = 0x40000000
    };
/** A synonym of CtError. */
typedef CtError CartoTypeError;

/** A result code typed as an integer. */
typedef uint32_t CtResult;

#endif
